﻿#pragma once
#include "assert.h"

//#pragma execution_character_set( "utf-8" )

// Forces debug messages to print in all builds
//#define PRINT_DEBUG

// Prints a message in the debugger in debug builds only
// TODO: Have different levels of log messages (Log, Warning, Error)
#if defined(_DEBUG) || defined(PRINT_DEBUG)
	#define DebugPrint(fmtstr, ...) DPrintf(fmtstr, __VA_ARGS__)
#else
	#define DebugPrint(fmtstr, ...)
#endif

// Prints a message in the debugger in all builds
#define PrintMsg(fmtstr, ...) DPrintf(fmtstr,  __VA_ARGS__)

namespace RTCam {
	wstring GetExePath();
	wstring GetCurPath();

	string StringFormat(_In_z_ const char* format, ...);
	wstring WstringFormat(_In_z_ const wchar_t* format, ...);

	// Prints a string with the given format to the debugger (and console, if it is enabled).
	void DPrintf(_In_z_ const char* format, ...);
	void DPrintf(_In_z_ const wchar_t* format, ...);

	// Checks whether a string is made of only whitespace
	bool IsWhitespace(string str);

	// Functions that nullify pointers after delete/release
	template<class T>
	inline void SafeDelete(T*& pVal) {
		delete pVal;
		pVal = nullptr;
	}

	template<class T>
	inline void SafeDeleteArray(T*& pVal) {
		delete[] pVal;
		pVal = nullptr;
	}

	template<class T>
	inline void SafeRelease(T*& pVal) {
		pVal->Release();
		pVal = nullptr;
	}

	// Clamps a value to a range
	template<typename T>
	inline T Clamp(T x, T min, T max) {
		return x < min ? min : (x > max ? max : x);
	}

	inline float ClampedLerp(float x0, float x1, float t) {
		if(t <= 0) return x0;
		if(t >= 1) return x1;

		float delta = x1 - x0;
		return x0 + (delta * t);
	}

	// Conversion between UTF-8 strings and UTF-16 wstrings
	inline wstring Utf8ToUtf16(_In_z_ const char* str) {
		// Treat nullptr as an error case
		ASSERT(str != nullptr);

		int bufferSize = MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, str, -1, nullptr, 0);
		if(bufferSize == 0) {
			// TODO: Report the specific error
			return wstring();
		}

		unique_ptr<wchar_t[]> buffer(new wchar_t[bufferSize]);
		MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, str, -1, buffer.get(), bufferSize);
		return wstring(buffer.get());
	}

	inline string Utf16ToUtf8(_In_z_ const wchar_t* wstr) {
		// Treat nullptr as an error case
		ASSERT(wstr != nullptr);
		
		int bufferSize = WideCharToMultiByte(CP_UTF8, WC_ERR_INVALID_CHARS, wstr, -1, nullptr, 0, nullptr, nullptr);
		if(bufferSize == 0) {
			// TODO: Report the specific error
			return string();
		}

		unique_ptr<char[]> buffer(new char[bufferSize]);
		WideCharToMultiByte(CP_UTF8, WC_ERR_INVALID_CHARS, wstr, -1, buffer.get(), bufferSize, nullptr, nullptr);
		return string(buffer.get());
	}

	inline void ShowMessageBox(_In_z_ const wchar_t* message, _In_z_ const wchar_t* windowTitle) {
		MessageBox(nullptr, message, windowTitle, MB_OK);
	}

	// Retrieve the system defined error message for the given HRESULT
	wstring GetErrorMessage(HRESULT hr);

	// Shows a message box with the error message, prints the error message to the console, 
	// and throws an exception. In debug builds, breaks before throwing the exception.
	inline void BreakAndThrow(_In_z_ const char* errorMsg, _In_z_ const wchar_t* errorMsgW) {
		PrintMsg(errorMsg);
		ShowMessageBox(errorMsgW, L"Error!");
		DebugBreak();
		throw std::exception(errorMsg);
	}
	inline void BreakAndThrow(_In_z_ const char* errorMsg) {
		wstring errorMsgW = Utf8ToUtf16(errorMsg);
		BreakAndThrow(errorMsg, errorMsgW.c_str());
	}
	inline void BreakAndThrow(_In_z_ const wchar_t* errorMsgW) {
		string errorMsg = Utf16ToUtf8(errorMsgW);
		BreakAndThrow(errorMsg.c_str(), errorMsgW);
	}
	
	// If the hr indicates a fail, show a message box describing the error and throw an exception.
	inline void ThrowIfFailed(HRESULT hr)
	{
		if (FAILED(hr)) {
			wstring errorMsgW = GetErrorMessage(hr);
			BreakAndThrow(errorMsgW.c_str());
		}
	}

	// Sets a D3D resource name string for debugging.
	// (From DirectXTK)
	template<size_t nameLength>
	inline void SetDebugObjectName(_In_ ID3D11DeviceChild* resource, _In_z_ const char (&name)[nameLength])
	{
#if defined(_DEBUG) || defined(PROFILE)
		resource->SetPrivateData(WKPDID_D3DDebugObjectName, nameLength - 1, name);
#else
		UNREFERENCED_PARAMETER(resource);
		UNREFERENCED_PARAMETER(name);
#endif
	}

	inline void SetDebugObjectName(_In_ ID3D11DeviceChild* resource, const string& name)
	{
#if defined(_DEBUG) || defined(PROFILE)
		resource->SetPrivateData(WKPDID_D3DDebugObjectName, name.length(), name.c_str());
#else
		UNREFERENCED_PARAMETER(resource);
		UNREFERENCED_PARAMETER(name);
#endif
	}
}

#define THROW_UNIMPLEMENTED_EXCEPTION() \
	do { \
		const char* shortFilename = (strrchr(__FILE__, '\\') + 1); \
		string errorMsg = StringFormat("Unimplemented: %s in %s on line %i\n", string(__FUNCSIG__), string(shortFilename), __LINE__); \
		PrintMsg(errorMsg.c_str()); \
		DebugBreak(); \
		throw std::exception(errorMsg.c_str()); \
	} while(0)